import React, { useEffect, useState, useCallback } from 'react';
import styled from 'styled-components';
import axios from 'axios';
import { getUID } from '../../../../Helper';
import { ChevronDown, Search } from 'lucide-react';

// Keep your original styled components and enhance them
const DropdownContainer = styled.div`
  position: relative;
  margin-bottom: 24px;
  flex: 1;
  max-width: 300px;
  width: 100%;
  
  @media (min-width: 768px) {
    margin-right: 16px;
  }
`;

const DropdownButton = styled.button`
  display: flex;
  align-items: center;
  justify-content: space-between;
  width: 100%;
  padding: 12px 16px;
  background: rgb(30, 35, 40);
  border: none;
  border-radius: 12px;
  color: #ffffff;
  cursor: pointer;
  font-size: 16px;
  font-weight: 600;
  box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
  transition: all 0.3s ease;

  &:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(0, 0, 0, 0.15);
  }
`;

const DropdownPanel = styled.div`
  position: absolute;
  top: 100%;
  left: 0;
  right: 0;
  background: rgb(30, 35, 40);
  border: none;
  border-radius: 12px;
  margin-top: 8px;
  max-height: 400px;
  overflow-y: auto;
  z-index: 1000;
  padding: 12px;
  box-shadow: 0 8px 25px rgba(0, 0, 0, 0.2);
`;

const SearchContainer = styled.div`
  position: relative;
  padding: 0px;
  border-radius: 8px;
  border: 1px solid rgb(58 65 66);
  margin-top: 12px;
`;

const SearchInput = styled.input`
  width: 100%;
  padding: 12px 12px 12px 40px;
  background: transparent;
  border: none;
  border-radius: 8px;
  color: #ffffff;
  font-size: 16px;
  &::placeholder {
    color: #fff;
  }
`;

const Input = styled.input`
  flex: 1;
  padding: 6px 10px;
  background: rgb(30, 35, 40);
  border: 1px solid rgba(255, 255, 255, 0.05);
  border-radius: 8px;
  color: #ffffff;
  font-size: 14px;
  &::placeholder {
    color: #fff;
  }
`;

const DepositButton = styled.div`
  margin-top: auto;
  flex: 1;
  padding: 7px 9px;
  display: flex;
  justify-content: center;
  align-items: center;
  background: #f09c06;
  color: white !important;
  border: none;
  border-radius: 0.5rem;
  font-weight: 600;
  cursor: pointer;
  
  &:hover {
    opacity: 0.9;
  }
`;

const SearchIcon = styled.div`
  position: absolute;
  left: 14px;
  top: 49%;
  transform: translateY(-50%);
  color: #fff;
`;

const CoinOption = styled.div`
  display: flex;
  align-items: center;
  padding: 14px;
  cursor: pointer;
  border-radius: 8px;
  transition: all 0.2s ease;
  &:hover {
    background: rgba(255, 255, 255, 0.1);
    transform: translateX(4px);
  }
  ${props => props.isSelected && `
    background: rgba(255, 255, 255, 0.15);
    font-weight: 600;
  `}
`;

const Container = styled.div`
  min-height: 100vh;
  background: #1a1f2b;
  padding: 16px;
  color: #ffffff;
  
  @media (min-width: 768px) {
    padding: 24px;
  }
`;

const Container1 = styled.div`
  color: white;
  padding: 24px;
  border-radius: 16px;
  width: 100%;
  max-width: 1200px;
  margin: 0 auto;
  background: #232935;
  box-shadow: 0 4px 20px rgba(0, 0, 0, 0.2);
`;

const TableHeader = styled.div`
  display: grid;
  grid-template-columns: repeat(4, 1fr);
  gap: 16px;
  color: #9ca3af;
  font-size: 14px;
  font-weight: 600;
  margin-bottom: 16px;
  padding: 12px 16px;
  background: rgba(255, 255, 255, 0.03);
  border-radius: 8px;
  
  @media (max-width: 640px) {
    display: none;
  }
`;

const TableRow = styled.div`
  display: grid;
  grid-template-columns: repeat(4, 1fr);
  gap: 16px;
  font-size: 14px;
  color: white;
  margin-bottom: 12px;
  padding: 16px;
  border-radius: 8px;
  transition: all 0.2s ease;
  
  &:hover {
    background: rgba(255, 255, 255, 0.05);
  }
  
  @media (max-width: 640px) {
    grid-template-columns: 1fr;
    background: rgba(255, 255, 255, 0.03);
    margin-bottom: 12px;
  }
`;

const EmptyState = styled.div`
  display: flex;
  flex-direction: column;
  align-items: center;
  justify-content: center;
  padding: 64px 16px;
  color: #9ca3af;
`;

const PlaceholderText = styled.div`
  color: #fff;
  font-size: 18px;
  margin-top: 16px;
  text-align: center;
`;

const StatusIndicator = styled.div`
  color: ${props => 
    props.status === 'Done' ? '#4CAF50' : 
    props.status === 'Success' ? '#4CAF50' :
    props.status === 'In Queue' ? '#FFC107' :
    props.status === 'Pending' ? '#FFC107' :
    props.status === 'Failed' ? '#F44336' : '#FFFFFF'
  };
  font-weight: 600;
  padding: 4px 12px;
  border-radius: 12px;
  background: rgba(255, 255, 255, 0.1);
  display: inline-block;
`;

const RefreshButton = styled.button`
  background: linear-gradient(145deg, #f09c06, #3BA364);
  border: none;
  border-radius: 12px;
  color: #ffffff;
  cursor: pointer;
  padding: 12px 20px;
  font-weight: 600;
  display: flex;
  align-items: center;
  gap: 8px;
  transition: all 0.3s ease;
  box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
  
  &:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(0, 0, 0, 0.15);
  }
`;

const TableContainer = styled.div`
  min-height: 300px;
  max-height: 500px;
  overflow-y: auto;
  background: #2a3140;
  border-radius: 12px;
  padding: 16px;
`;

const HeaderContainer = styled.div`
  display: flex;
  flex-direction: column;
  align-items: center;
  gap: 16px;
  margin-bottom: 24px;
  
  @media (min-width: 768px) {
    flex-direction: row;
    justify-content: space-between;
    align-items: center;
  }
`;

const MobileLabel = styled.span`
  display: none;
  color: #9ca3af;
  font-weight: 600;
  margin-right: 8px;
  
  @media (max-width: 640px) {
    display: inline;
  }
`;

const SkylaTransaction = () => {
  const [activeTab, setActiveTab] = useState("Deposit");
  const [openDropDown, setOpenDropDown] = useState(false);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [transactionData, setTransactionData] = useState([]);
  const [refreshTrigger, setRefreshTrigger] = useState(0);

  useEffect(() => {
    const handleClickOutside = (event) => {
      if (!event.target.closest(".dropdown-container")) {
        setOpenDropDown(false);
      }
    };

    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, []);

  const fetchData = useCallback(async () => {
    setLoading(true);
    setError(null);

    try {
      console.log("Fetching data for:", activeTab);
      const response = await axios.get(
        `https://api.boss707.com/api/${activeTab.toLowerCase()}New`,
        { params: { uid:getUID} }
      );

      console.log("API Response:", response.data);

      let processedData = [];

      if (activeTab === "Withdraw") {
        const withdrawals = response.data.withdrawals || [];
        const fiatWithdrawals = response.data.fiatWithdrawals || [];
        
        const formattedFiatWithdrawals = fiatWithdrawals.map(item => ({
          date: item.date,
          amount: `${item.amount} ${item.currency || ""}`,
          status: item.status,
          details: `Bank: ${item.bank_name || "N/A"}, Acc: ${item.account_number || "N/A"}`,
          type: "Fiat"
        }));
        
        const formattedWithdrawals = withdrawals.map(item => ({
          date: item.date || item.created_at,
          amount: item.amount,
          status: item.status,
          details: item.details,
          type: "Crypto"
        }));
        
        processedData = [...formattedWithdrawals, ...formattedFiatWithdrawals];
      } else {
        if (Array.isArray(response.data)) {
          processedData = response.data.map(item => ({
            date: item.date || item.created_at,
            amount: item.amount,
            status: item.status,
            details: item.details,
            type: "Crypto"
          }));
        } else {
          const deposits = response.data.deposits || [];
          const fiatDeposits = response.data.fiatDeposits || [];
          
          const formattedDeposits = deposits.map(item => ({
            date: item.date || item.created_at,
            amount: item.amount,
            status: item.status,
            details: item.details,
            type: "Crypto"
          }));
          
          const formattedFiatDeposits = fiatDeposits.map(item => ({
            date: item.date || item.created_at,
            amount: `${item.amount} ${item.currency || ""}`,
            status: item.status,
            details: item.details,
            type: "Fiat"
          }));
          
          processedData = [...formattedDeposits, ...formattedFiatDeposits];
        }
      }

      const sortedData = processedData.sort((a, b) => {
        return new Date(b.date) - new Date(a.date);
      });

      setTransactionData(sortedData);
    } catch (err) {
      console.error("API call failed", err);
      setError("Failed to fetch data.");
      setTransactionData([]);
    } finally {
      setLoading(false);
    }
  }, [activeTab, refreshTrigger]);

  useEffect(() => {
    fetchData();
  }, [fetchData]);

  const formatDateTime = (dateString) => {
    if (!dateString) return "N/A";
    return new Date(dateString).toLocaleDateString(undefined, {
      year: "numeric",
      month: "short",
      day: "numeric",
      hour: "2-digit",
      minute: "2-digit",
    });
  };

  const handleRefresh = () => {
    setRefreshTrigger(prev => prev + 1);
  };

  return (
    <Container>
      <Container1>
        <HeaderContainer>
          <DropdownContainer className="dropdown-container">
            <DropdownButton
              onClick={(e) => {
                e.stopPropagation();
                setOpenDropDown(!openDropDown);
              }}
            >
              {activeTab}
              <ChevronDown size={20} />
            </DropdownButton>

            {openDropDown && (
              <DropdownPanel onClick={(e) => e.stopPropagation()}>
                {["Deposit", "Withdraw"].map((type) => (
                  <CoinOption
                    key={type}
                    onClick={() => {
                      if (activeTab !== type) {
                        setTransactionData([]);
                        setActiveTab(type);
                      }
                      setOpenDropDown(false);
                    }}
                    isSelected={activeTab === type}
                  >
                    {type}
                  </CoinOption>
                ))}
              </DropdownPanel>
            )}
          </DropdownContainer>
          
          <RefreshButton onClick={handleRefresh}>
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
              <path d="M21 2v6h-6"></path>
              <path d="M3 12a9 9 0 0 1 15-6.7L21 8"></path>
              <path d="M3 22v-6h6"></path>
              <path d="M21 12a9 9 0 0 1-15 6.7L3 16"></path>
            </svg>
            Refresh
          </RefreshButton>
        </HeaderContainer>

        {loading ? (
          <EmptyState>
            <div className="spinner" style={{ width: "40px", height: "40px", borderRadius: "50%", border: "3px solid rgba(255,255,255,0.1)", borderTopColor: "#f09c06", animation: "spin 1s linear infinite" }}></div>
            <PlaceholderText>Loading data...</PlaceholderText>
            <style>{`
              @keyframes spin {
                0% { transform: rotate(0deg); }
                100% { transform: rotate(360deg); }
              }
            `}</style>
          </EmptyState>
        ) : error ? (
          <EmptyState>
            <div style={{ display: "flex", flexDirection: "column", justifyContent: "center", alignItems: "center", height: "300px", width: "100%" }}>
              <img src="/assets/images/bcgames/affiliate/empty.webp" alt="No transactions" style={{ height: "150px", width: "150px", objectFit: "contain" }} />
              <PlaceholderText>Oops! Failed to fetch data!</PlaceholderText>
              <RefreshButton onClick={handleRefresh} style={{ marginTop: "16px" }}>
                Try Again
              </RefreshButton>
            </div>
          </EmptyState>
        ) : transactionData.length === 0 ? (
          <EmptyState>
            <div style={{ display: "flex", flexDirection: "column", justifyContent: "center", alignItems: "center", height: "300px", width: "100%" }}>
              <img src="/assets/images/bcgames/affiliate/empty.webp" alt="No transactions" style={{ height: "150px", width: "150px", objectFit: "contain" }} />
              <PlaceholderText>No {activeTab.toLowerCase()} transactions found</PlaceholderText>
            </div>
          </EmptyState>
        ) : (
          <TableContainer>
            <TableHeader>
              <div>Date</div>
              <div>Amount</div>
              <div>Status</div>
              <div>{activeTab === "Deposit" ? "Transaction" : "Details"}</div>
            </TableHeader>

            {transactionData.map((transaction, index) => (
              <TableRow key={index}>
                <div>
                  <MobileLabel>Date:</MobileLabel>
                  {formatDateTime(transaction.date)}
                </div>
                <div>
                  <MobileLabel>Amount:</MobileLabel>
                  {transaction.amount}
                  {transaction.type && <span style={{ marginLeft: "4px", fontSize: "12px", opacity: "0.7" }}>({transaction.type})</span>}
                </div>
                <div>
                  <MobileLabel>Status:</MobileLabel>
                  <StatusIndicator status={transaction.status}>
                    {transaction.status}
                  </StatusIndicator>
                </div>
                <div style={{ wordBreak: "break-word" }}>
                  <MobileLabel>{activeTab === "Deposit" ? "Transaction" : "Details"}:</MobileLabel>
                  {activeTab === "Deposit" ? transaction.details : 'N/A'}
                </div>
              </TableRow>
            ))}
          </TableContainer>
        )}
      </Container1>
    </Container>
  );
};

export default SkylaTransaction;